import * as THREE from 'https://unpkg.com/three@v0.150.0/build/three.module.js';

// The usual three
var renderer, scene, camera;

// Here we will store all the data that we send to the shader
// Everything except faces are four channel aligned
var vertices = [];
var faces = [];
var colors = [];
var normals = [];
var triangleCount = 0;
var vertexCount = 0;
var vertexShader, fragmentShader;
var dataTextures; // We use 2 data textures: 0 for vertices, 1 for face data (normal and color)

var chopperBladesStartIndex, chopperBladesEndIndex; // Store indices of the vertices into those variables
var chopperBladesStartFaceIndex, chopperBladesEndFaceIndex; // Store the indices of the faces into these variables

var clock = new THREE.Clock(); //We use this to get the time delta

function toRad(degree) {
    return Math.PI * 2 * degree / 360;
}

window.onLoad = async () => {
    var canvasContainer = document.getElementById('myCanvasContainer');
    var width = 200;
    var height = 200;

    vertexShader = await (await fetch('vertex.glsl')).text();
    fragmentShader = await (await fetch('fragment.glsl')).text();

    renderer = new THREE.WebGLRenderer();
    renderer.setSize(width, height);
    canvasContainer.appendChild(renderer.domElement);

    // These will create the data for the hangar and the chopper
    addHangar();
    addChopper();

    // Specify some positions for our scene, here we need a viewer position, because we will be working in the world space
    var lightPosition = new THREE.Vector3(0.0, 10.0, 5.0);
    var viewerPosition = new THREE.Vector3(0.0, 0.0, 24.0);

    // This will generate our textures
    dataTextures = generateDataTextures();

    var uniforms = {
        vertexData: {
            type: 't',
            value: dataTextures[0]
        },
        triangleData: {
            type: 't',
            value: dataTextures[1]
        },
        faces: {
            type: 'iv',
            value: faces // In here we have the face indices
        },
        lightPosition: {
            type: 'v3',
            value: lightPosition
        },
        viewerPosition: {
            type: 'v3',
            value: viewerPosition
        }
    };

    // In the shaders we will replace the triangleCount and vertexCount
    // The shaders will be compiled with fixed values for them
    fragmentShader = fragmentShader.replace(/\(\(triangleCount\)\)/g, triangleCount);
    fragmentShader = fragmentShader.replace(/\(\(vertexCount\)\)/g, vertexCount);

    var material = new THREE.ShaderMaterial({
        uniforms: uniforms,
        vertexShader: vertexShader,
        fragmentShader: fragmentShader
    });

    // From here on, we put together the Three.js scene for rendering the raytracer

    /**
     * --Task--
     * -Create a plane aligned with the camera's view. Use width and height 55 for the plane.
     * -Create a corresponding Orthographic camera that is looking exactly that plane.
     */

    // draw(); // Comment that in, when there is something to render and something to render with. :)
}

var dt;
function draw() {
    requestAnimationFrame(draw);

    dt = clock.getDelta();
    updateChopper(dt); // In this function we update the chopper geometry data

    renderer.render(scene, camera);
}

/**
 * Hangar consists of one cuboid
 */
function addHangar() {
    createCube(
        new THREE.Vector3(0, 0, 0),    // Position
        new THREE.Vector3(-50, 50, 50),// Scale
        new THREE.Color(0x222244)      // Color
    );
}

/**
 * Chopper consists or 3 cuboids
 */
function addChopper() {
    createCube(
        new THREE.Vector3(0, -15, 0),
        new THREE.Vector3(10, 4, 4),
        new THREE.Color(0x2222ff)
    );

    // --Task-- store the current index of the vertices and the current index of the faces here

    createCube(
        new THREE.Vector3(5.5, -12, 0),
        new THREE.Vector3(10, 0.6, 1),
        new THREE.Color(0x22ff22)
    );
    createCube(
        new THREE.Vector3(-5.5, -12, 0),
        new THREE.Vector3(10, 0.6, 1),
        new THREE.Color(0x22ff22)
    );

    // --Task-- store the current index of the vertices and the current index of the faces here
    // Because we need to keep track, where are our blade vertices and normals in the data.
}

/**
 * This function creates a sphere and adds its data to the arrays.
 * You may also try this function out. Although a sphere will create many triangles to the scene.
 */
function createSphere(position, scale, color) {
    if (scale == undefined) {
        scale = new THREE.Vector(1, 1, 1);
    }

    if (color == undefined) {
        color = new THREE.Color(0xff0000);
    }

    var geometry = new THREE.SphereGeometry(8, 8, 8);
    geometry.applyMatrix4(new THREE.Matrix4().set(
        scale.x, 0.0, 0.0, position.x,
        0.0, scale.y, 0.0, position.y,
        0.0, 0.0, scale.z, position.z,
        0.0, 0.0, 0.0, 1.0
    ));

    addObject(geometry, color);
}

/**
 * This function creates a cube and adds its data to the arrays.
 * Used by hangar and chopper.
 */
function createCube(position, scale, color) {
    if (scale == undefined) {
        scale = new THREE.Vector(1, 1, 1);
    }

    if (color == undefined) {
        color = new THREE.Color(0xff0000);
    }

    var geometry = new THREE.BoxGeometry(1, 1, 1);
    geometry.applyMatrix4(new THREE.Matrix4().set(
        scale.x, 0.0, 0.0, position.x,
        0.0, scale.y, 0.0, position.y,
        0.0, 0.0, scale.z, position.z,
        0.0, 0.0, 0.0, 1.0
    ));

    addObject(geometry, color);
}

/**
 * --Task--
 * Finish this function.
 * This function takes the correct information from the geometry and adds it to the arrays.
 * Keep in mind that vertices, normals, and colors have to be 4 channel aligned
 */
function addObject(geometry, color) {
    var currentFaceIndex = vertices.length / 4;
    // Each object has their own faces indices. But if we store them all in a data array, we need to change the numbering.

    // Go through the vertices of the geometry (geometry.attributes.position.array) and add the values to the "vertices" array

    // Have Three.js calculate the normals for us
    geometry.computeVertexNormals();

    // TODO
    // Go through the faces of the geometry (geometry.index.array) and add:
    // - Face indices, don't forget to change them
    // - Normals of the vertices of faces (geometry.attributes.normal.array)
    // - Colors, add the same color for all vertices (unless you want to change it?)


    // After this, we have increased our data
    triangleCount += geometry.index.count / 3;
    vertexCount += geometry.attributes.position.count;
}

/**
 * This function generates the data textures to be sent to the shaders.
 * There are 2 data textures:
 * -First for storing the vertex coordinates
 * -Second for storing the face data (normal and color)
 *
 * --Task--
 * Finish this function.
 */
function generateDataTextures() {
    var data = new Float32Array(vertexCount * 4); // This is a sequential memory array in JS
    var index;

    // We copy the data from the vertices to the data array
    for (var i = 0; i < vertexCount * 4; i++) {
        // Vertex position
        data[i] = vertices[i];
    }

    // Create a new data texture, one big row. NN filter, no mipmaps.
    var vertexTexture = new THREE.DataTexture(data, vertexCount, 1, THREE.RGBAFormat, THREE.FloatType);
    vertexTexture.minFilter = THREE.NearestFilter;
    vertexTexture.magFilter = THREE.NearestFilter;
    vertexTexture.generateMipmaps = false;
    vertexTexture.needsUpdate = true;

    /*
     * --Task--
     * We want to do a similar thing for the normals and colors.
     * For this, we create a triangleCount X 2 data texture. First row will store the normals, second will store the colors.
     */
    var rowLength = triangleCount * 4;
    var triangleData = new Float32Array(rowLength * 2);

    // Populate the triangleData with the correct data
    // Create the DataTexture with that data, configure the filtering as before.
    // You can specify triangleTexture.flipY = true, in order to start from the top

    console.log('The "triangleTexture" is undefined until you create it here');
    return [vertexTexture, triangleTexture]; // This will give an error until you create a triangleTexture.
}

/**
 * This function updates the chopper geometry, ie rotates the blades.
 *
 * --Task--
 * Finish this function.
 */
function updateChopper(dt) {
    var vertex = new THREE.Vector3();
    var normal = new THREE.Vector3();
    var up = new THREE.Vector3(0, 1, 0);
    var rotation = 2 * dt;

    // Use the indices we stored, when creating the chopper, to get a correct segment of the data.
    // Create THREE.Vector3() objects from the data, rotate them, put the data back.
    // When you have a Vector3, a function applyAxisAngle can be used to rotate a vector.

    // Do the same for both the vertices of the blades and...

    // ... normals of the blades.
    // Don't forget to tell the dataTexture that it needs an update (data has changed).
}