// ---------------------------- Includes -------------------------- //
#include <stdlib.h>             // C++ standard library
#include <stack>                // We use the standard C++ stack implementation to create model matrix stacks
#include <thread>               // Threading
#include <stdio.h>              // Input/Output
#include <GL/glew.h>          // OpenGL Extension Wrangler -
#include <GLFW/glfw3.h>         // Windows and input
#include <glm/glm.hpp>          // OpenGL math library
#include <glm/gtx/rotate_vector.hpp>     // Vector rotation
#include "shader_util.h"        // Utility methods to keep this file a bit shorter.

// --------------- Forward declarations ------------- //

void addHangar();
void addChopper();
void addCube(glm::vec3 position, glm::vec3 scale, glm::vec3 color);
void addObject(std::vector<glm::vec3> objectVertices, std::vector<glm::ivec3> objectFaces, std::vector<glm::vec3> objectNormals, std::vector<glm::vec3> objectColors, glm::mat3 modelMatrix);
void generateDataTextures();
void updateChopper();

//Here we will store the VBO and data texture indices
GLuint planeVBO;
GLuint vertexDataTex, triangleDataTex;

//Here we will store the data for the data textures
std::vector<glm::vec3> vertices;
std::vector<glm::ivec3> faces;
std::vector<glm::vec3> colors;
std::vector<glm::vec3> normals;

//Keep track of the triangle and vertex counts
int triangleCount = 0;
int vertexCount = 0;

//We need to know, where are the blades in our data
int chopperBladesStartIndex, chopperBladesEndIndex;
int chopperBladesStartFaceIndex, chopperBladesEndFaceIndex;

//Light and viewer positions
glm::vec3 lightPosition = glm::vec3(0.0, 10.0, 5.0);
glm::vec3 viewerPosition = glm::vec3(0.0, 0.0, 24.0);

#define CONCAT_PATHS(A, B) A "/" B
#define ADD_ROOT(B) CONCAT_PATHS(TASK_ROOT_PATH, B)

// --- Load the shaders declared in glsl files in the project folder ---//
shader_prog rayShader(ADD_ROOT("shaders/ray.vert.glsl"), ADD_ROOT("shaders/ray.frag.glsl"));

//For the time delta
float lastUpdateTime = glfwGetTime();

/**
 * This function generates the data textures with the correct data.
 *
 * --Task--
 * Finish this function.
 */
void generateDataTextures() {

    //First we create a texture for the vertices.
    //Populating this data is easy, because data in std::vector<glm::vec3> is already sequential
    glGenTextures(1, &vertexDataTex);
    glBindTexture(GL_TEXTURE_2D, vertexDataTex);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB32F, vertexCount, 1, 0, GL_RGB, GL_FLOAT, &vertices[0].x);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    /**
     * Next create a data texture with the size "triangleCount X 2".
     * First row will store the normals, second row will store the colors.
     */
    int rowLength = triangleCount * 3;
    auto triangleData = std::vector<float>(triangleCount * 3 * 2); //We need sequential memory

    //Loop through the sequential memory and populate it with correct values

    /* You may want to debug with:
    for (int i = 0; i < triangleCount * 3; i++) {
        printf("%f, ", triangleData[i]);
    }
    for (int i = 0; i < triangleCount * 3; i++) {
        printf("%f, ", triangleData[rowLength + i]);
    }
    */

    //Generate the correct texture triangleDataTex with correct data
}

/**
 * This function applies a transformation on the object and adds it to the global arrays.
 *
 * --Task--
 * Finish this function
 */
void addObject(std::vector<glm::vec3> objectVertices, std::vector<glm::ivec3> objectFaces, std::vector<glm::vec3> objectNormals, std::vector<glm::vec3> objectColors, glm::mat3 modelMatrix, glm::vec3 position) {

    /*
     * First loop through the vertices:
     * - Apply the model matrix
     * - Add the position
     * - You can debug with: printf("Vertex: %f, %f, %f \n", objectVertices[i].x, objectVertices[i].y, objectVertices[i].z);
     */

    /*
     * Create a normal matrix from the model matrix.
     * Find the correct start index for the faces.
     * Loop through the faces:
     * - Apply the normalMatrix for the normals.
     * - Multiply the normals with the scaleSign (this is a hack to create the hangar by flipping the x axis.
     * - Add correct face index to the faces std::vector.
     */
    int scaleSign = ceil(abs(modelMatrix[0][0] * modelMatrix[1][1] * modelMatrix[2][2]) / (modelMatrix[0][0] * modelMatrix[1][1] * modelMatrix[2][2]));

    //Loop through the faces here

    //Vertices, normals and colors we add as is
    vertices.insert(vertices.end(), objectVertices.begin(), objectVertices.end());
    normals.insert(normals.end(), objectNormals.begin(), objectNormals.end());
    colors.insert(colors.end(), objectColors.begin(), objectColors.end());

    /* Can debug with:
    for (int i = 0; i < faces.size(); i++) {
        printf("f%d: %d, %d, %d\n", i, faces[i].x, faces[i].y, faces[i].z);
    }
    */

    //Increase the vertexCount and triangleCount here
}

/**
 * Hangar consists of one cuboid. We scale with a negative x coefficient to flip the normals.
 */
void addHangar() {
    addCube(glm::vec3(0.0f, 0.0f, 0.0f), glm::vec3(-50.0f, 50.0f, 50.0f), glm::vec3(0.13, 0.13, 0.26));
}

/**
 * Chopper consists of three cuboids.
 *
 * --Task--
 * Get the correct indices for the blades in this function.
 */
void addChopper() {
    addCube(glm::vec3(0.0f, -15.0f, 0.0f), glm::vec3(10.0f, 4.0f, 4.0f), glm::vec3(0.2, 0.2, 1.0));

    //Assign chopperBladesStartIndex and chopperBladesStartFaceIndex here

    addCube(glm::vec3(5.5f, -12.0f, 0.0f), glm::vec3(10.0f, 0.6f, 1.0f), glm::vec3(0.2, 1.0, 0.2)); //0.8
    addCube(glm::vec3(-5.5f, -12.0f, 0.0f), glm::vec3(10.0f, 0.6f, 1.0f), glm::vec3(0.2, 1.0, 0.2));

    //Assign the chopperEndIndex and the chopperEndFaceIndex here
}

/**
 * This function will rotate the vertex positions and the normals for the blades.
 *
 * --Task--
 * Finish this function.
 */
void updateChopper() {
    float rotation = float(glm::radians(fmod((glfwGetTime() - lastUpdateTime) * 100.0, 360.0)));
    lastUpdateTime = glfwGetTime(); //We have the delta, store the current time

    //Go through the correct vertices in the "vertices" array and apply rotation with glm::rotateY on them

    //Make the vertex data texture active, we currently /know/ that it is in GL_TEXTURE0.
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, vertexDataTex);

    //Use glTexSubImage2D command the change the corresponding data

    //Go through the correct normals in the "normals" array and apply rotation with glm::rotateY on them

    //Again we /know/ that the face data is stored in GL_TEXTURE1
    glActiveTexture(GL_TEXTURE1);
    glBindTexture(GL_TEXTURE_2D, triangleDataTex);

    //Use glTexSubImage2D command the change the corresponding data
}

/**
 * This function:
 * - creates a cube geometry (vertices, faces, normals)
 * - creates a vector of colors for the faces (same color for all faces, unless you want to change that)
 * - creates a correct scaling model matrix
 * - calls addObject with that data
 */
void addCube(glm::vec3 position, glm::vec3 scale, glm::vec3 color) {
    std::vector<glm::vec3> cubeVertices = std::vector<glm::vec3>();
    std::vector<glm::ivec3> cubeFaces = std::vector<glm::ivec3>();
    std::vector<glm::vec3> cubeNormals = std::vector<glm::vec3>();
    std::vector<glm::vec3> cubeColors = std::vector<glm::vec3>();

    float elements[9] = {
          scale.x, 0.0f, 0.0f,
          0.0f, scale.y, 0.0f,
          0.0f, 0.0f, scale.z,
    };
    glm::mat3 modelMatrix = glm::make_mat3(elements);

    cubeVertices.push_back(glm::vec3(-0.5, -0.5, 0.5));
    cubeVertices.push_back(glm::vec3(0.5, -0.5, 0.5));
    cubeVertices.push_back(glm::vec3(0.5,  0.5, 0.5));
    cubeVertices.push_back(glm::vec3(-0.5,  0.5, 0.5));
    cubeVertices.push_back(glm::vec3(-0.5, -0.5, -0.5));
    cubeVertices.push_back(glm::vec3(0.5, -0.5, -0.5));
    cubeVertices.push_back(glm::vec3(0.5,  0.5, -0.5));
    cubeVertices.push_back(glm::vec3(-0.5, 0.5, -0.5));

    cubeFaces.push_back(glm::ivec3(0, 1, 2)); //Front
    cubeFaces.push_back(glm::ivec3(0, 2, 3));
    cubeFaces.push_back(glm::ivec3(1, 5, 6)); //Right
    cubeFaces.push_back(glm::ivec3(1, 6, 2));
    cubeFaces.push_back(glm::ivec3(4, 6, 5)); //Back
    cubeFaces.push_back(glm::ivec3(4, 7, 6));
    cubeFaces.push_back(glm::ivec3(0, 3, 4)); //Left
    cubeFaces.push_back(glm::ivec3(4, 3, 7));
    cubeFaces.push_back(glm::ivec3(3, 6, 7)); //Top
    cubeFaces.push_back(glm::ivec3(3, 2, 6));
    cubeFaces.push_back(glm::ivec3(0, 4, 1)); //Bottom
    cubeFaces.push_back(glm::ivec3(1, 4, 5));

    cubeNormals.push_back(glm::vec3(0.0, 0.0, 1.0)); //Front
    cubeNormals.push_back(glm::vec3(0.0, 0.0, 1.0));
    cubeNormals.push_back(glm::vec3(1.0, 0.0, 0.0)); //Right
    cubeNormals.push_back(glm::vec3(1.0, 0.0, 0.0));
    cubeNormals.push_back(glm::vec3(0.0, 0.0, -1.0)); //Back
    cubeNormals.push_back(glm::vec3(0.0, 0.0, -1.0));
    cubeNormals.push_back(glm::vec3(-1.0, 0.0, 0.0)); //Left
    cubeNormals.push_back(glm::vec3(-1.0, 0.0, 0.0));
    cubeNormals.push_back(glm::vec3(0.0, 1.0, 0.0)); //Top
    cubeNormals.push_back(glm::vec3(0.0, 1.0, 0.0));
    cubeNormals.push_back(glm::vec3(0.0, -1.0, 0.0)); //Bottom
    cubeNormals.push_back(glm::vec3(0.0, -1.0, 0.0));

    for (int i = 0; i < cubeFaces.size(); i++) {
        cubeColors.push_back(color);
    }

    addObject(cubeVertices, cubeFaces, cubeNormals, cubeColors, modelMatrix, position);
};



// ---------------------------- Input -------------------------- //
// This method is called when keyboard event happens.
// Sets GLFW window should close flag to true, when escape key is pressed.
static void key_callback(GLFWwindow* window, int key, int scancode, int action, int mods) {
    if (action == GLFW_PRESS) {
        switch(key) {
            case GLFW_KEY_ESCAPE:
                glfwSetWindowShouldClose(window, GL_TRUE);
            break;
        }
    }
}

// ---------------------------- Main -------------------------- //
int main(int argc, char *argv[]) {
    GLFWwindow *win;

    if (!glfwInit()) {
        exit (EXIT_FAILURE);
    }

    win = glfwCreateWindow(250, 250, "Ray Chopper!", NULL, NULL);

    if (!win) {
        glfwTerminate();
        exit(EXIT_FAILURE);
    }

    glfwMakeContextCurrent(win);

    //Some extensions are marked as experimental. To use the latest
    //version of OpenGL supported, we have to set this flag to true.
    glewExperimental = GL_TRUE;
    GLenum status = glewInit();

    if(status != GLEW_OK) {
        fprintf(stderr, "Error: %s\n", glewGetErrorString(status));
    }

    const GLubyte* renderer = glGetString (GL_RENDERER); // get renderer string
    const GLubyte* version = glGetString (GL_VERSION); // version as a string
    printf ("Renderer: %s\n", renderer);
    printf ("OpenGL version supported %s\n", version);

    glfwSetKeyCallback(win, key_callback);

    //We add our hangar and the chopper
    addHangar();
    addChopper();

    //Then we generate the data textures
    generateDataTextures();

    //Print the current counts
    printf("Vertices: %d \n", vertexCount);
    printf("Faces: %d \n", triangleCount);

    //We replace the counts in the textures, compiling them with the fixed count values
    rayShader.replaceInt("((vertexCount))" , std::max(1, vertexCount));
    rayShader.replaceInt("((triangleCount))" , std::max(1, triangleCount)); // max is here to avoid a shader compilation error
    rayShader.use(); //Compiling

    //Shader needs those variables
    rayShader.uniform3vec("lightPosition", lightPosition);   //Light position
    rayShader.uniform3vec("viewerPosition", viewerPosition); //Viewer position
    rayShader.uniformVectorIVec3("faces", faces);            //Array of ivec3-s that have the face indices in it
    rayShader.uniformTex2D("vertexData", vertexDataTex);     //Vertex data texture
    rayShader.uniformTex2D("triangleData", triangleDataTex); //Triangle (face) data texture


    /**
     * --Task--
     * Create a plane (square) with the width = height = 55.
     * Create an orthographic projection that looks exactly at that plane.
     */
    float planeSizeHalf = 55.0f / 2.0f;

    //Use std::vector<glm::vec3> to store the plane vertices.
    //Most easiest is to use a TRIANGLE_FAN to represent that geometry.

    /* Then bind those positions to the VBO using:
        glGenVertexArrays(1, &planeVBO);
        glBindVertexArray(planeVBO);
        rayShader.attributeVectorVec3("position", planePoints);
    */

    //Create an orthographic projection with glm::ortho and glm::lookAt.
    /* Send the matrices to the shaders using:
        rayShader.uniformMatrix4fv("projectionMatrix", projection);
        rayShader.uniformMatrix4fv("viewMatrix", view);
    */

    glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
    glEnable(GL_TEXTURE_2D); //We enable the texturing

    while (!glfwWindowShouldClose(win)) {
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        //First we update the chopper (rotate the blades)
        updateChopper();

        //Then we bind the planeVBO and draw it as a triangle fan
        glBindVertexArray(planeVBO);
        glDrawArrays(GL_TRIANGLE_FAN, 0, 4);


        glfwSwapBuffers(win);
        glfwPollEvents();
        std::this_thread::sleep_for(std::chrono::milliseconds(1));
    }

    glfwTerminate();
    exit(EXIT_SUCCESS);
    return 0;
}
